<?php

namespace App\Models;

use App\Http\Traits\HasSearch;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Coupon extends Model
{
    use HasFactory, SoftDeletes, HasSearch;

    protected $searchable = [
        'title',
        'body',
        'code'
    ];
    protected $guarded = ['id'];
    protected $casts = [
        'active' => 'boolean',
        'used' => 'boolean',
        'on_delivery_cost' => 'boolean',
        'for_marketing' => 'boolean'
    ];
    protected $appends = ['valid_now'];

    public function getValidNowAttribute(): bool
    {
        return $this->active && !$this->used && Carbon::now()->isBefore(Carbon::parse($this->expiration));
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function scopeForMarketing($query, $forMarketing)
    {
        return $query->when($forMarketing !== null, function ($query) use ($forMarketing) {
            $query->where('for_marketing', '=', $forMarketing);
        });
    }

    public function scopeExpired($query, $expired)
    {
        if ($expired)
            return $query->when($expired !== null, function ($query) use ($expired) {
                if (boolval($expired))
                    $query->whereDate('expiration', '<', Carbon::now()->toDateTimeString());
                else
                    $query->whereDate('expiration', '>=', Carbon::now()->toDateTimeString());
            });
    }

    public function scopeValidNow($query, $valid)
    {
        return $query->when($valid !== null, function ($query) use ($valid) {
            if (boolval($valid))
                $query->where('active', '=', '1')
                    ->where('used', '!=', '0')
                    ->whereDate('expiration', '>', Carbon::now()->toDateString());
            else
                $query
                    ->where('active', '!=', '1')
                    ->orWhere('used', '=', '0')
                    ->orWhereDate('expiration', '<=', Carbon::now()->toDateString());
        });
    }
}
