<?php

namespace App\Http\Requests;

use App\Rules\Mobile;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Support\Facades\Response;
use Illuminate\Validation\Rule;

class SellerRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules(): array
    {
        $validation = [
            'email' => [
                'required',
                'email',
                Rule::unique('users')->where('deleted', 0)
            ],
            'phone' => [
                'required', new Mobile()
                , Rule::unique('users')->where('deleted', 0)
            ],
            'first_name' => [
                'required'
                , 'max:50'
            ],
            'last_name' => [
                'required'
                , 'max:50'
            ],
            'image' => [
                'nullable'
            ],
            'city_id' => [
                'required'
                , Rule::exists('world_cities', 'id')
            ],

            'supplier_since' => ['required', 'date_format:Y-m-d'],
            'corporation_name' => ['required', 'max:50'],
            'longitude' => ['nullable'],
            'latitude' => ['nullable'],
            'description_en' => ['nullable'],
            'description_ar' => ['nullable'],
            'address_en' => ['nullable'],
            'address_ar' => ['nullable'],
        ];
        if ($this->getMethod() === 'POST') {
            return $validation;
        }
        return array_merge($validation, [
            'email' => [
                'required',
                'email',
                'max:100', Rule::unique('users')->where('deleted', 0)->ignore(request()->route('id'))
            ],
            'phone' => ['required'
                , 'max:50', new Mobile()
                , Rule::unique('users')->where('deleted', 0)->ignore(request()->route('id'))
            ],
            'activated' => ['required', 'boolean']
        ]);
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(Response::failure($validator->errors()->first()));
    }
}
