<?php

namespace App\Http\Requests;

use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Support\Facades\Response;
use Illuminate\Validation\Rule;

class ProductRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules(): array
    {
        if ($this->getMethod() === 'PATCH') {
            return [
                'best_collection' => [
                    'required',
                    'boolean'
                ],
                'newly_added' => [
                    'required',
                    'boolean'
                ],
                'flash_deal' => [
                    'required',
                    'boolean'
                ],
                'selected' => [
                    'required',
                    'boolean'
                ]
            ];
        }
        $validation = [
            'short_description_en' => [
                'required'
            ],
            'short_description_ar' => [
                'required'
            ],
            'description_en' => [
                'required'
            ],
            'description_ar' => [
                'required'
            ],
            'meta_tag_title_en' => [
                'required'
            ],
            'meta_tag_title_ar' => [
                'required'
            ],
            'meta_tag_keywords_en' => [
                'required',
                'array'
            ],
            'meta_tag_keywords_ar' => [
                'required',
                'array'
            ],
            'tags' => [
                'required',
                'array'
            ],
            'seo' => [
                'required',
                'array'
            ],
            'sku' => [
                'required'
            ],
            'location' => [
                'required'
            ],
            'latitude' => [
                'nullable'
            ],
            'longitude' => [
                'nullable'
            ],
            'price' => [
                'required',
                'numeric'
            ],
            'quantity' => [
                'required',
                'numeric'
            ],
            'min_quantity' => [
                'required',
                'integer',
                'min:1'
            ],
            'max_quantity' => [
                'nullable',
                'integer'
            ],
            'subtract_stock' => [
                'required',
                'boolean'
            ],
            'shipping_timing' => [
                'nullable'
            ],
            'requires_shipping' => [
                'required',
                'boolean'
            ],
            'availability_date' => [
                'required',
                'date_format:Y-m-d'
            ],
            'length_class' => [
                'nullable',
                Rule::in(['Millimeter', 'Inch', 'Centimeter'])
            ],
            'dimensions' => [
                'nullable',
                'array'
            ],
            'dimensions.length' => [
                'nullable',
                'numeric'
            ],
            'dimensions.width' => [
                'nullable',
                'numeric'
            ],
            'dimensions.height' => [
                'nullable',
                'numeric'
            ],
            'weight_class' => [
                'nullable',
                Rule::in(['Gram', 'Kilogram', 'Pound', 'Ounce'])
            ],
            'weight' => [
                'nullable',
                'numeric'
            ],
            'activated' => [
                'required',
                'boolean'
            ],
            'best_collection' => [
                'required',
                'boolean'
            ],
            'newly_added' => [
                'required',
                'boolean'
            ],
            'flash_deal' => [
                'required',
                'boolean'
            ],
            'selected' => [
                'required',
                'boolean'
            ],
            'delivery_estimation' => [
                'nullable'
            ],
            'sort' => [
                'required',
                'integer'
            ],
            'video' => [
                'nullable'
            ],
            'image' => [
                'nullable'
            ],
            'images' => [
                'nullable',
                'array'
            ],
            'required_points' => [
                'required',
                'integer'
            ],

            'categories' => [
                'required',
                'array'
            ],
            'categories.*' => [
                'nullable',
                'integer',
                /*Rule::exists('product_categories', 'id')->whereNull('deleted_at')*/
            ],
            'brand_id' => [
                'required',
                'integer',
                /*Rule::exists('brands', 'id')->whereNull('deleted_at')*/
            ],
            'discount' => [
                'nullable'
            ],
            'discount.quantity' => [
                'nullable',
                'integer'
            ],
            'discount.price' => [
                'nullable',
                'numeric'
            ],
            'discount.percentage' => [
                'nullable',
                'numeric',
                'gt:0'
            ],
            'discount.start_date' => [
                'nullable',
                'date_format:Y-m-d H:i:s'
            ],
            'discount.end_date' => [
                'nullable',
                'date_format:Y-m-d H:i:s',
                'after_or_equal:discount.*.start_date'
            ],
            'special_offer' => [
                'nullable'
            ],
            'special_offer.price' => [
                'nullable',
                'numeric'
            ],
            'special_offer.percentage' => [
                'nullable',
                'numeric',
                'gt:0'
            ],
            'special_offer.start_date' => [
                'nullable',
                'date_format:Y-m-d H:i:s'
            ],
            'special_offer.end_date' => [
                'nullable',
                'date_format:Y-m-d H:i:s',
                'after_or_equal:special_offer.*.start_date'
            ],
            'out_of_stock_id' => [
                'required',
                'integer',
                /*Rule::exists('out_of_stocks', 'id')->whereNull('deleted_at')*/
            ],
            'available_delivery_options' => [
                'required',
                'array'
            ],
            'available_delivery_options.*' => [
                'required',
                'integer',
                /*Rule::exists('delivery_options', 'id')->whereNull('deleted_at')*/
            ],
            'attributes' => [
                'nullable',
                'array',
                'max:30'
            ],
            'attributes.*' => [
                'nullable',
                'array'
            ],
            'attributes.*.attribute_id' => [
                'required',
                'integer',
                /*Rule::exists('attributes', 'id')*/
            ],
            'attributes.*.value_en' => [
                'required'
            ],
            'attributes.*.value_ar' => [
                'required'
            ],
            'associations' => [
                'nullable',
                'max:5'
            ],
            'associations.*.association_id' => [
                'required',
                'integer',
                'distinct',
                /*Rule::exists('associations', 'id')*/
            ],
            'attributes.*.products' => [
                'nullable',
                'array',
                'max:10'
            ],
            'attributes.*.products.*' => [
                'required',
                'integer',
                'distinct'
            ],
            'available_payment' => [
                'required',
                'array'
            ],
            'available_payment.*' => [
                'required',
                'integer',
                /*Rule::exists('payments', 'id')->whereNull('deleted_at')*/
            ],
            'seller_id' => [
                'nullable',
                'integer',
                Rule::exists('sellers', 'id')
            ]
            /*'options' => [
                'nullable',
                'array'
            ],
            'options.*' => [
                'nullable',
                'array'
            ],
            'options.*.option_id' => [
                'nullable',
                'integer',
                Rule::exists('options', 'id')
            ],
            'options.*.required' => [
                'nullable',
                'boolean'
            ],
            'options.*.product_option_values' => [
                'nullable',
                'array'
            ],
            'options.*.product_option_values.*' => [
                'nullable',
                'array'
            ],
            'options.*.product_option_values.*.option_value_id' => [
                'nullable',
                'integer',
                Rule::exists('option_values', 'id')
            ],
            'options.*.product_option_values.*.quantity' => [
                'nullable',
                'integer'
            ],
            'options.*.product_option_values.*.subtract_stock' => [
                'nullable',
                'boolean'
            ],
            'options.*.product_option_values.*.price_diff' => [
                'nullable',
                'numeric'
            ],
            'options.*.product_option_values.*.required_points_diff' => [
                'nullable',
                'integer'
            ],
            'options.*.product_option_values.*.weight_diff' => [
                'nullable',
                'numeric'
            ],
            'available_payment' => [
                'required',
                'array'
            ],
            'available_payment.*' => [
                'required',
                'integer',
                Rule::exists('payments', 'id')->whereNull('deleted_at')
            ],
            'tax_class_id' => [
                'required',
                'integer',
                Rule::exists('tax_classes', 'id')
            ],
            'product_type_id' => [
                'required',
                'integer',
                Rule::exists('product_types', 'id')->whereNull('deleted_at')
            ],
            'seller_id' => [
                'nullable',
                'integer',
                Rule::exists('sellers', 'id')
            ]
            */

        ];
        if ($this->getMethod() === 'POST') {
            return array_merge($validation, [
                'name_en' => [
                    'required'
                    , 'different:name_ar', Rule::unique('products')->where('deleted', 0)
                ],
                'name_ar' => [
                    'required'
                    , 'different:name_en', Rule::unique('products')->where('deleted', 0)],
                'code' => [
                    'required',
                    Rule::unique('products')->where('deleted', 0)
                ],
                'barcode' => [
                    'required',
                    Rule::unique('products')->where('deleted', 0)
                ],
                'colors' => [
                    'nullable',
                    'array'
                ],
                'colors.*' => [
                    'nullable',
                    'array'
                ],
                'colors.*.name_en' => [
                    'required',
                    'max:100'
                ],
                'colors.*.name_ar' => [
                    'required',
                    'max:100'
                ],
                'colors.*.hex' => [
                    'nullable'
                ],
                'colors.*.image' => [
                    'required'
                ],
                'sizes' => [
                    'nullable',
                    'array'
                ],
                'sizes.*' => [
                    'nullable',
                    'array'
                ],
                'sizes.*.name' => [
                    'required',
                    'max:100'
                ],
                'sizes.*.image' => [
                    'nullable'
                ],
            ]);
        }
        return array_merge($validation, [
            'name_en' => [
                'required'
                , 'different:name_ar', Rule::unique('products')->where('deleted', 0)
                    ->ignore(request()->route('id'))
            ],
            'name_ar' => [
                'required'
                , 'different:name_en', Rule::unique('products')->where('deleted', 0)
                    ->ignore(request()->route('id'))
            ],
            'code' => [
                'required',
                Rule::unique('products')->where('deleted', 0)
                    ->ignore(request()->route('id'))
            ],
            'barcode' => [
                'required',
                Rule::unique('products')->where('deleted', 0)
                    ->ignore(request()->route('id'))
            ],
        ]);
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(Response::failure($validator->errors()->first()));
    }
}
