<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\UserRequest;
use App\Models\User;
use App\Services\CouponService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Khsing\World\Models\City;

class UserController extends Controller
{
    private $couponService;

    public function __construct(CouponService $couponService)
    {
        $this->couponService = $couponService;
        if (config('app.middlewares_enabled') === true) {
            $this->middleware('auth:api');
            $this->middleware('has_permission:view_user')->only(['index', 'export', 'show']);
            $this->middleware('has_permission:create_user')->only(['store']);
            $this->middleware('has_permission:update_user')->only(['update']);
            $this->middleware('has_permission:delete_user')->only(['delete']);
        }
    }

    public function index(): JsonResponse
    {
        return Response::success(
            User::search(request()->query('word_filter'))
                ->phoneFilter(request()->query('phone_filter'))
                ->active(request()->query('activated'))
                ->roleFilter(request()->query('role_filter'))
                ->where('role', '!=', 'Admin')
                ->where('role', '!=', 'Super Admin')
        );
    }

    public function show($id): JsonResponse
    {
        return Response::success(
            User::query()->with(['city', 'appPreference'])->find($id)
        );
    }

    public function store(UserRequest $request): JsonResponse
    {
        DB::beginTransaction();
        $userInput = $request->validated();
        $userInput['activated'] = '1';
        $userInput['timezone'] = City::query()->find($request->input('city_id'))->country->timezone;
        $userInput['role'] = 'Member';
        $user = User::query()->create($userInput);
        $this->couponService->assignExistingCoupons($user);
        DB::commit();

        return Response::success($user->load(['city', 'appPreference']));

    }

    public function update(UserRequest $request, $id): JsonResponse
    {
        //DB::enableQueryLog();
        $user = User::query()
            ->findOrFail($id);
        $userInput = $request->validated();

        $userInput['timezone'] = City::query()->find($request->input('city_id'))->country->timezone;
        $user->update($userInput);
        return Response::success($user->load(['city', 'appPreference']));
        //return Response::success(DB::getQueryLog());
    }

    public function destroy($id): JsonResponse
    {
        DB::beginTransaction();
        User::query()->findOrFail($id)->delete();
        DB::commit();
        return Response::success(null);

    }
}
